# -*- coding: utf-8 -*-

import odoo.addons.decimal_precision as dp
from odoo.exceptions import UserError

from odoo import api, fields, models


class AccountAssetImpairment(models.TransientModel):
    _name = 'account.asset.impairment'

    def _get_asset(self):
        asset_id = self.env.context.get('active_id', False)
        return asset_id

    @api.model
    @api.depends('asset')
    def _get_book_value(self):
        if not self.asset:
            return 0
        else:
            depreciation_amount = sum(self.asset.depreciation_move_ids.filtered(lambda r: r.state == 'posted').mapped('amount_total'))
            self.asset_bv = self.asset.original_value - depreciation_amount

    def _get_default_impairment_account(self):
        return self.env.company.id.impairment_loss

    @api.depends('asset')
    def _get_original_value(self):
        for record in self:
            record.asset_cost = record.asset.original_value

    asset = fields.Many2one(comodel_name='account.asset', string='Asset', default=_get_asset, readonly=True, required=True)
    impairment_date = fields.Date(string='Impairment Date', required=True)
    accounting_date = fields.Date(string='Accounting Date', required=True)
    asset_cost = fields.Float(compute='_get_original_value', string='Asset Cost', readonly=True)
    asset_bv = fields.Float(string='Asset Book Value', readonly=True, compute='_get_book_value')
    amount = fields.Float(string='New Impaired Value', digits=dp.get_precision('Account'))
    impairment_loss_account = fields.Many2one(comodel_name='account.account', string='Impairment Loss Account',
                                              default=_get_default_impairment_account)
    journal = fields.Many2one(comodel_name='account.journal', string='Journal', required=True)

    @api.onchange('amount')
    def onchange_amount(self):
        if self.amount > self.asset_bv:
            raise UserError('Impaired amount cannot be greater than current book value')
        if self.asset.depreciation_move_ids.filtered(lambda x: x.state != 'posted'):
            raise UserError('You must delete draft depreciation journals before processing impairment.')

    def move_line(self, asset, account_id, debit, credit, journal, expense=False):
        move_line = {
            'name': asset.name,
            'account_id': account_id,
            'debit': debit,
            'credit': credit,
            'journal_id': journal,
            'partner_id': False,
            'analytic_account_id': False,
            'currency_id': False,
            'amount_currency': 0.0
        }
        return move_line

    def get_journal_id(self):
        return self.env['account.asset.default'] \
            .search([('company_id', '=', self.env.user.company_id.id)])[0].invoice_journal.id

    def button_process(self):

        if self.env.user.company_id.fiscalyear_lock_date > self.accounting_date:
            raise UserError('This accounting date is before your posting cut off date')

        asset_id = self.env.context.get('active_id', False)
        asset = self.env['account.asset'].browse(asset_id)

        # check draft moves posted and delete depreciation lines not linked to a draft or posted move
        lines_to_check = [x for x in asset.depreciation_line_ids if x.move_check]
        for line in lines_to_check:
            if line.move_id and line.move_id.state == 'draft':
                raise UserError('There is a draft journal created for this asset. Please either delete or post.')

        if self.amount < 0:
            raise UserError('Cannot impair to negative value.')

        impairment_amount = self.asset_bv - self.amount
        move_lines = []
        # impairment  expense
        line = self.move_line(asset, self.impairment_loss_account.id, impairment_amount, 0, self.journal, True)

        move_lines.append((0, 0, line))
        # Accum Depn amount
        line = self.move_line(asset, asset.account_depreciation_id.id, 0,
                              impairment_amount, self.journal, False)
        move_lines.append((0, 0, line))

        move_vals = {
            'ref': asset.code,
            'name': asset.name + ' - Impairment',
            'date': self.accounting_date or False,
            'journal_id': self.journal,
            'line_ids': move_lines,
        }

        self.env['account.move'].sudo().create(move_vals)

        # update the asset
        note = self.asset.note
        if note:
            note = note + "\ Impairment processed on {date} changing asset value from {old_bv} to {new_value}".format(
                date=self.impairment_date,
                old_bv=self.asset_bv,
                new_value=self.amount)
        else:
            note = "Impairment processed on {date} changing asset value from {old_bv} to {new_value}".format(
                date=self.impairment_date,
                old_bv=self.asset_bv,
                new_value=self.amount)
        asset.write({
            'value_residual': self.amount,
            'state': 'open',
            'note': note})

        asset.message_post(body='Impairment Processed '
                                'Impairment processed on {date} changing asset value'
                                ' from {old_bv} to {new_value} by {user}'.format(
            date=self.impairment_date, old_bv=self.asset_bv, new_value=self.amount, user=self.env.user.name))
