# -*- coding: utf-8 -*-


import base64
from datetime import date
from io import BytesIO as StringIO

import xlsxwriter

from odoo import models, fields


class AccountAssetTaxReport(models.TransientModel):
    _name = 'account.asset.tax.report'

    report_name = fields.Char(size=64, string='Report Name', default="Asset Tax Report.xlsx")
    as_at_date = fields.Date(string='As At Date', required=True)
    data = fields.Binary(string='Download File', readonly=True)
    lines = fields.One2many(comodel_name='account.asset.tax.report.line',
                            inverse_name='wizard_id', string="Lines")

    def diff_month(self, asset_date, year_start_date):
        months = 0
        if year_start_date.year > asset_date.year:
            months = year_start_date.year - asset_date.year * 12
        months += year_start_date.month - asset_date.month
        if months < 0:
            months = 0

        return months

    def calculate_year_start_date(self):
        self.ensure_one()
        year_end_month = int(self.env.company.fiscalyear_last_month)
        if year_end_month == 12:
            month = 1
        else:
            month = year_end_month + 1

        if self.as_at_date.month > year_end_month:
            year = self.as_at_date.year
        else:
            year = self.as_at_date.year - 1

        return date(year, month, 1)

    def format_date(self, as_at_date):
        day = as_at_date.day
        if day < 10:
            day = '0' + str(as_at_date.day)
        else:
            day = str(as_at_date.day)

        month = as_at_date.month
        if month < 10:
            month = '0' + str(as_at_date.month)
        else:
            month = str(as_at_date.month)

        formatted_date = day + '/' + month + '/' + str(as_at_date.year)
        return formatted_date

    def build_data(self):
        self.ensure_one()

        # Clear out existing.
        self.lines.unlink()

        wizard = self
        asset_ids = self.env['account.asset'].search([('company_id', '=', self.env.company.id), ('state', 'not in', ['draft', 'model'])])
        for asset in asset_ids:
            addition = opening_balance = disposal = depn_opening = depreciation = withdrawn = 0
            sold = False
            purchase_prior_year_start = False
            year_start_date = self.calculate_year_start_date()

            if asset.acquisition_date > wizard.as_at_date:
                continue
            if asset.disposal_date and asset.disposal_date < year_start_date:
                continue
            if asset.acquisition_date < year_start_date:
                opening_balance = asset.original_value
                purchase_prior_year_start = True
            else:
                addition = asset.original_value
            if asset.disposal_date and year_start_date <= asset.disposal_date <= wizard.as_at_date:
                disposal = asset.original_value
                sold = True

            """
            for NZ tax depreciation the rules are:
                1. if an asset is owned more than 6 months in the year, then a full year, else 6 months
                2. SL or DV
                3. will not allow for residual as only applies for some very specific assets
                4. difference in gain/loss on dispoal between accounting and tax handled by user
            """

            if purchase_prior_year_start:
                months_purchased_prior_report_start_date = self.diff_month(year_start_date, asset.acquisition_date)
                part_year = months_purchased_prior_report_start_date % 12
                if part_year and part_year < 6:
                    months_purchased_prior_report_start_date = months_purchased_prior_report_start_date - (months_purchased_prior_report_start_date % 12) + 6
                elif part_year:
                    months_purchased_prior_report_start_date = months_purchased_prior_report_start_date - (months_purchased_prior_report_start_date % 12) + 12
                months_this_report = self.diff_month(wizard.as_at_date, year_start_date)
            elif asset.acquisition_date >= year_start_date:
                months_purchased_prior_report_start_date = 0
                months_this_report = self.diff_month(wizard.as_at_date, asset.acquisition_date)
                if months_this_report >= 6:
                    months_this_report = self.diff_month(wizard.as_at_date, year_start_date)
                else:
                    months_this_report = 6
            if sold:
                months_this_report = self.diff_month(wizard.as_at_date, asset.disposal_date)

            if asset.tax_method == 'degressive':  # DV
                years = int(months_purchased_prior_report_start_date / 12)
                dv = asset.original_value
                for x in range(0, years):
                    depn = dv * asset.tax_rate / 100
                    depn_opening += depn
                    if depn_opening > asset.original_value:
                        depn_opening = asset.original_value
                        break
                    dv = dv - depn
                depreciation = dv / 12 * months_this_report * asset.tax_rate / 100
            else:
                depn_opening = asset.original_value * months_purchased_prior_report_start_date / 12 * asset.tax_rate / 100
                if depn_opening > asset.original_value:
                    depn_opening = asset.original_value
                opening_bv = asset.original_value - depn_opening
                if opening_bv:
                    depreciation = months_this_report / 12 * asset.original_value * asset.tax_rate / 100
                    if depreciation > opening_bv:
                        depreciation = opening_bv
                else:
                    depreciation = 0.0

            if sold:
                withdrawn = depn_opening
                depreciation = 0.0

            self.env['account.asset.tax.report.line'].create({
                'wizard_id': self.id,
                'asset_id': asset.id,
                'asset_category_id': asset.account_asset_id.id,
                'cost_price_opening': opening_balance,
                'addition': addition,
                'disposal': disposal,
                'depn_opening': depn_opening,
                'depreciation': depreciation,
                'withdrawn': withdrawn,

            })

    def button_process(self):
        """ Create the report. """
        wizard = self.browse(self.ids[0])
        data = StringIO()

        # build the data table
        self.build_data()

        workbook = xlsxwriter.Workbook(data)
        worksheet = workbook.add_worksheet('Data')
        format_number = workbook.add_format({'num_format': '#,##0.00', 'align': 'right'})
        format_row = workbook.add_format({'text_wrap': True, 'bold': True, 'size': 12})
        format_row_right = workbook.add_format({'text_wrap': True, 'bold': True, 'size': 12, 'align': 'right'})
        format_row_no_wrap = workbook.add_format({'bold': True})
        format_row_centre = workbook.add_format({'bold': True, 'align': 'center'})
        format_total_cell = workbook.add_format({'bold': True, 'num_format': '#,##0.00', 'align': 'right'})

        row = 0

        date_for_report = self.format_date(wizard.as_at_date)

        # write report heading
        worksheet.set_column('J:J', 3)
        worksheet.set_column('O:O', 3)
        worksheet.write(0, 0, 'Asset Report - Accounting - As At ' + date_for_report, format_row_no_wrap)
        row += 1

        worksheet.merge_range(row, 5, row, 8, '<------------------ Cost Price ----------------->', format_row_centre)
        worksheet.merge_range(row, 10, row, 13, '<---------------- Depreciation --------------->', format_row_centre)
        row += 1

        worksheet.write(row, 0, 'Code', format_row)
        worksheet.write(row, 1, 'Description', format_row)
        worksheet.write(row, 2, 'Serial #', format_row)
        worksheet.write(row, 3, 'Purchase Date', format_row)
        worksheet.write(row, 4, 'Disposal Date', format_row)
        worksheet.write(row, 5, 'Opening Balance', format_row_right)
        worksheet.write(row, 6, 'Additions', format_row_right)
        worksheet.write(row, 7, 'Disposals', format_row_right)
        worksheet.write(row, 8, 'Closing Balance', format_row_right)
        worksheet.write(row, 10, 'Opening Balance', format_row_right)
        worksheet.write(row, 11, 'Depreciation', format_row_right)
        worksheet.write(row, 12, 'Withdrawn', format_row_right)
        worksheet.write(row, 13, 'Closing Balance', format_row_right)
        worksheet.write(row, 15, 'Book Value', format_row_right)
        row += 1

        asset_opening_total = asset_addition_total = asset_disposal_total \
            = depn_opening_total = depn_depreciation_total = depn_withdrawn_total = 0

        categories = list(set([x.asset_category_id for x in self.lines]))
        for category in categories:
            row += 2
            worksheet.write(row, 0, category.name, format_row_no_wrap)
            row += 1
            asset_opening_cat_total = asset_addition_cat_total = asset_disposal_cat_total \
                = depn_opening_cat_total = depn_depreciation_cat_total = depn_withdrawn_cat_total = 0
            assets = [x for x in self.lines if x.asset_category_id == category]
            for asset in assets:
                if asset.asset_id.disposal_date:
                    disposal_date = self.format_date(asset.asset_id.disposal_date)
                else:
                    disposal_date = ''
                worksheet.write(row, 0, asset.asset_id.code)
                worksheet.write(row, 1, asset.asset_id.name)
                worksheet.write(row, 2, asset.asset_id.serial_number or '')
                worksheet.write(row, 3, self.format_date(asset.asset_id.acquisition_date))
                worksheet.write(row, 4, disposal_date or '')
                worksheet.write(row, 5, asset.cost_price_opening, format_number)
                worksheet.write(row, 6, asset.addition, format_number)
                worksheet.write(row, 7, asset.disposal, format_number)
                worksheet.write(row, 8, asset.cost_price_opening + asset.addition - asset.disposal, format_number)
                worksheet.write(row, 10, asset.depn_opening, format_number)
                worksheet.write(row, 11, asset.depreciation, format_number)
                worksheet.write(row, 12, asset.withdrawn, format_number)
                worksheet.write(row, 13, asset.depn_opening + asset.depreciation - asset.withdrawn, format_number)
                worksheet.write(row, 15, (asset.cost_price_opening + asset.addition - asset.disposal) -
                                (asset.depn_opening + asset.depreciation - asset.withdrawn), format_number)
                row += 1

                asset_opening_cat_total += asset.cost_price_opening
                asset_addition_cat_total += asset.addition
                asset_disposal_cat_total += asset.disposal
                depn_opening_cat_total += asset.depn_opening
                depn_depreciation_cat_total += asset.depreciation
                depn_withdrawn_cat_total += asset.withdrawn

                asset_opening_total += asset.cost_price_opening
                asset_addition_total += asset.addition
                asset_disposal_total += asset.disposal
                depn_opening_total += asset.depn_opening
                depn_depreciation_total += asset.depreciation
                depn_withdrawn_total += asset.withdrawn

            worksheet.write(row, 5, asset_opening_cat_total, format_total_cell)
            worksheet.write(row, 6, asset_addition_cat_total, format_total_cell)
            worksheet.write(row, 7, asset_disposal_cat_total, format_total_cell)
            worksheet.write(row, 8, asset_opening_cat_total + asset_addition_cat_total - asset_disposal_cat_total,
                            format_total_cell)
            worksheet.write(row, 10, depn_opening_cat_total, format_total_cell)
            worksheet.write(row, 11, depn_depreciation_cat_total, format_total_cell)
            worksheet.write(row, 12, depn_withdrawn_cat_total, format_total_cell)
            worksheet.write(row, 13, depn_opening_cat_total + depn_depreciation_cat_total - depn_withdrawn_cat_total,
                            format_total_cell)
            worksheet.write(row, 15, asset_opening_cat_total + asset_addition_cat_total - asset_disposal_cat_total -
                            (depn_opening_cat_total + depn_depreciation_cat_total - depn_withdrawn_cat_total),
                            format_total_cell)
            row += 2

        worksheet.write(row, 1, 'Grand Total', format_row_no_wrap)
        worksheet.write(row, 5, asset_opening_total, format_total_cell)
        worksheet.write(row, 6, asset_addition_total, format_total_cell)
        worksheet.write(row, 7, asset_disposal_total, format_total_cell)
        worksheet.write(row, 8, asset_opening_total + asset_addition_total - asset_disposal_total, format_total_cell)
        worksheet.write(row, 10, depn_opening_total, format_total_cell)
        worksheet.write(row, 11, depn_depreciation_total, format_total_cell)
        worksheet.write(row, 12, depn_withdrawn_total, format_total_cell)
        worksheet.write(row, 13, depn_opening_total + depn_depreciation_total - depn_withdrawn_total, format_total_cell)
        worksheet.write(row, 15, asset_opening_total + asset_addition_total - asset_disposal_total -
                        (depn_opening_total + depn_depreciation_total - depn_withdrawn_total), format_total_cell)

        worksheet.set_column('A:A', 8)
        worksheet.set_column('B:B', 30)
        worksheet.set_column('C:C', 8)
        worksheet.set_column('D:I', 14)
        worksheet.set_column('K:N', 14)
        worksheet.set_column('P:P', 14)

        workbook.close()
        data.seek(0)
        output = base64.encodestring(data.read())
        self.write({'data': output})  # maybe need self in brackets?

        return {
            'type': 'ir.actions.act_window',
            'res_model': 'account.asset.tax.report',
            'view_mode': 'form',
            'view_type': 'form',
            'res_id': wizard.id,
            'target': 'new', }


class account_asset_tax_report_line(models.TransientModel):
    _name = 'account.asset.tax.report.line'

    wizard_id = fields.Many2one(string='Wizard', comodel_name='account.asset.tax.report')
    asset_id = fields.Many2one(string='Asset', comodel_name='account.asset')
    asset_category_id = fields.Many2one(string="Category", comodel_name='account.account')
    cost_price_opening = fields.Float(string='Cost Price Opening')
    addition = fields.Float(string='Addition')
    disposal = fields.Float(string='Disposal')
    depn_opening = fields.Float(string='Opening Depreciation')
    depreciation = fields.Float(string='Depreciation')
    withdrawn = fields.Float('Withdrawn')

    _order = 'asset_category_id, asset_id'
