# -*- coding: utf-8 -*-

from datetime import datetime, timedelta

from dateutil.relativedelta import relativedelta

import odoo.addons.decimal_precision as dp
from odoo import api, fields, models
from odoo.exceptions import UserError


class AccountAssetImpairment(models.TransientModel):
    _name = 'account.asset.impairment'

    def _get_asset(self):
        asset_id = self.env.context.get('active_id', False)
        return asset_id

    @api.model
    @api.depends('asset')
    def _get_book_value(self):
        if not self.asset:
            return 0
        else:
            posted_depn_lines = self.env['account.asset.depreciation.line'].search(
                [('asset_id', '=', self.asset.id), ('move_posted_check', '=', True)])
            posted_depn = sum([x.amount for x in posted_depn_lines])
            bv = self.asset.value - posted_depn
            self.asset_bv = bv

    def _get_default_impairment_account(self):
        company = self.env.user.company_id.id
        default_rec = self.env['account.asset.default'].search([('company_id', '=', company)])[0]
        return default_rec.impairment_loss

    asset = fields.Many2one(comodel_name='account.asset.asset', string='Asset', default=_get_asset, readonly=True)
    impairment_date = fields.Date(string='Impairment Date', required=True)
    accounting_date = fields.Date(string='Accounting Date', required=True)
    asset_cost = fields.Float(string='Asset Cost', readonly=True, related='asset.value')
    asset_bv = fields.Float(string='Asset Book Value', readonly=True, compute='_get_book_value')
    amount = fields.Float(string='New Impaired Value', digits=dp.get_precision('Account'))
    impairment_loss_account = fields.Many2one(comodel_name='account.account', string='Impairment Loss Account',
                                              default=_get_default_impairment_account)
    depreciation_periods = fields.Integer(string='Remaining Depreciation Periods', required=True)

    @api.multi
    @api.onchange('amount')
    def onchange_amount(self):
        if self.amount > self.asset_bv:
            raise UserError('Impaired amount cannot be greater than current book value')

        lines_to_check = [x for x in self.asset.depreciation_line_ids if x.move_check]
        for line in lines_to_check:
            if line.move_id and line.move_id.state == 'draft':
                raise UserError('There is a draft journal created for this asset. Please either delete or post.')

        self.depreciation_periods = len([x.id for x in self.asset.depreciation_line_ids if not x.move_check])

    @api.multi
    def move_line(self, asset, account_id, debit, credit, journal, expense=False):
        move_line = {
            'name': asset.name,
            'account_id': account_id,
            'debit': debit,
            'credit': credit,
            'journal_id': journal,
            'partner_id': False,
            'analytic_account_id': False,
            'currency_id': False,
            'amount_currency': 0.0
        }
        return move_line

    def get_journal_id(self):
        return self.env['account.asset.default'] \
            .search([('company_id', '=', self.env.user.company_id.id)])[0].invoice_journal.id

    @api.multi
    def button_process(self):

        if self.env.user.company_id.fiscalyear_lock_date > self.accounting_date:
            raise UserError('This accounting date is before your posting cut off date')

        journal = self.env['account.asset.default'] \
            .search([('company_id', '=', self.env.user.company_id.id)])[0].journal.id
        if not journal:
            raise UserError('No journal set up')

        asset_id = self.env.context.get('active_id', False)
        asset = self.env['account.asset.asset'].browse(asset_id)

        # check draft moves posted and delete depreciation lines not linked to a draft or posted move
        lines_to_check = [x for x in asset.depreciation_line_ids if x.move_check]
        for line in lines_to_check:
            if line.move_id and line.move_id.state == 'draft':
                raise UserError('There is a draft journal created for this asset. Please either delete or post.')

        lines_to_delete = [x for x in asset.depreciation_line_ids if not x.move_check]
        for line in lines_to_delete:
            line.unlink()

        if self.amount < 0:
            raise UserError('Cannot impair to negative value.')

        if self.amount and not self.depreciation_periods:
            raise UserError('You need to specify the number of periods for depreciation.')

        highest_depn_line = self.env['account.asset.depreciation.line'].search([('asset_id', '=', self.asset.id)],
                                                                               order='sequence desc', limit=1)
        latest_date_line = self.env['account.asset.depreciation.line'].search([('asset_id', '=', self.asset.id)],
                                                                              order='depreciation_date desc', limit=1)
        if highest_depn_line:
            next_sequence = highest_depn_line.sequence + 1
        else:
            next_sequence = 1
        seq_for_impairment = next_sequence
        next_sequence += 1
        if latest_date_line:
            depreciation_date = fields.Date.from_string(latest_date_line.depreciation_date) + relativedelta(months=1)
        else:
            year = datetime.today().year
            month = datetime.today().month
            depreciation_date_str = '{year}-{month}-01'.format(year=year, month=month)
            depreciation_date_first = fields.Date.from_string(depreciation_date_str) + relativedelta(months=1)
            depreciation_date = depreciation_date_first - timedelta(days=1)

        depreciated_value = self.asset_bv
        monthly_depreciation = round((self.amount / self.depreciation_periods), 2)
        amount_to_depreciate = self.amount
        while amount_to_depreciate > monthly_depreciation:
            amount_to_depreciate -= monthly_depreciation
            depreciated_value += monthly_depreciation
            self.env['account.asset.depreciation.line'].create({
                'asset_id': self.asset.id,
                'name': self.asset.name,
                'sequence': next_sequence,
                'move_check': False,
                'depreciation_date': fields.Date.to_string(depreciation_date),
                'amount': monthly_depreciation,
                'move_posted_check': False,
                'remaining_value': amount_to_depreciate,
                'depreciated_value': depreciated_value,
                'category_id': self.asset.category_id.id
            })
            next_sequence += 1
            # always make depreciation date end of month so cannot just add a month
            first_of_month = '{year}-{month}-01'.format(year=depreciation_date.year, month=depreciation_date.month)
            first_of_next_month = fields.Date.from_string(first_of_month) + relativedelta(months=2)
            depreciation_date = first_of_next_month - timedelta(days=1)

        if amount_to_depreciate:
            self.env['account.asset.depreciation.line'].create({
                'asset_id': self.asset.id,
                'name': self.asset.name,
                'sequence': next_sequence,
                'move_check': False,
                'depreciation_date': depreciation_date,
                'amount': amount_to_depreciate,
                'move_posted_check': False,
                'remaining_value': 0,
                'depreciated_value': self.amount,
                'category_id': self.asset.category_id.id
            })
            next_sequence += 1

        impairment_amount = self.asset_bv - self.amount
        move_lines = []
        # impairment  expense
        line = self.move_line(asset, self.impairment_loss_account.id, impairment_amount, 0, journal, True)

        move_lines.append((0, 0, line))
        # Accum Depn amount
        line = self.move_line(asset, asset.category_id.account_depreciation_id.id, 0,
                              impairment_amount, journal, False)
        move_lines.append((0, 0, line))

        move_vals = {
            'ref': asset.code,
            'name': asset.name + ' - Impairment',
            'date': self.accounting_date or False,
            'journal_id': journal,
            'line_ids': move_lines,
        }
        move_id = self.env['account.move'].sudo().create(move_vals)

        # write the impairment amount as a depreciation line with type of impairment
        self.env['account.asset.depreciation.line'].create({
            'asset_id': self.asset.id,
            'name': self.asset.name,
            'sequence': seq_for_impairment,
            'depreciation_date': self.impairment_date,
            'amount': impairment_amount,
            'move_posted_check': False if move_id.state == 'draft' else True,
            'move_id': move_id.id,
            'move_check': True,
            'remaining_value': self.amount,
            'depreciated_value': self.asset_cost - self.amount,
            'line_type': 'impairment',
            'category_id': self.asset.category_id.id
        })

        # update the asset
        posted_depn = sum([x.amount for x in asset.depreciation_line_ids if x.move_posted_check])
        note = self.asset.note
        if note:
            note = note + "\ Impairment processed on {date} changing asset value from {old_bv} to {new_value}".format(
                date=self.impairment_date,
                old_bv=self.asset_bv,
                new_value=self.amount)
        else:
            note = "Impairment processed on {date} changing asset value from {old_bv} to {new_value}".format(
                date=self.impairment_date,
                old_bv=self.asset_bv,
                new_value=self.amount)
        asset.write({
            'value_residual': self.amount,
            'state': 'open',
            'note': note})

        asset.message_post(body='Impairment Processed '
                                'Impairment processed on {date} changing asset value'
                                ' from {old_bv} to {new_value} by {user}'.format(
            date=self.impairment_date, old_bv=self.asset_bv, new_value=self.amount, user=self.env.user.name))
