# -*- coding: utf-8 -*-
from odoo import api, fields, models
import odoo.addons.decimal_precision as dp
from odoo.exceptions import UserError


class AccountAssetDisposal(models.TransientModel):
    _name = 'account.asset.disposal'

    ################################################################################
    # Field's methods
    ################################################################################

    def get_default_rec(self, company):
        if not self.env['account.asset.default'].search([('company_id', '=', company)]):
            raise UserError('Please set up assets default in configuration')
        return self.env['account.asset.default'].search([('company_id', '=', company)])[0]

    def _get_default_capital_account(self):
        company = self.env.user.company_id.id
        default_rec = self.get_default_rec(company)
        return default_rec.capital_gain_account

    def _get_default_loss_account(self):
        company = self.env.user.company_id.id
        default_rec = self.get_default_rec(company)
        return default_rec.loss_on_disposal_account

    def _get_default_gain_account(self):
        company = self.env.user.company_id.id
        default_rec = self.get_default_rec(company)
        return default_rec.gain_on_disposal_account

    ################################################################################
    # Fields
    ################################################################################

    asset = fields.Many2one(comodel_name='account.asset.asset', string='Asset', readonly=True)
    disposal_date = fields.Date(string='Disposal Date', required=True)
    accounting_date = fields.Date(string='Accounting Date', required=True)
    tr_type = fields.Selection(string='Type', required=True, selection=[('sale', 'Sale'), ('write_off', 'Write-Off')],
                               help='A write-off has no proceeds to account for')
    asset_cost = fields.Float(string='Asset Cost')
    asset_bv = fields.Float(string='Asset Book Value')
    amount = fields.Float(string='Amount', digits=dp.get_precision('Account'))
    partner = fields.Many2one(comodel_name='res.partner', string='Sold To')
    create_invoice = fields.Boolean(string='Create Invoice',
                                    help='If ticked, Default Sale Income Account is'
                                         ' credited for sale amount in disposal journal.')
    capital_account = fields.Many2one(comodel_name='account.account', string='Capital Gain Account',
                                      default=_get_default_capital_account)
    loss_on_disposal_account = fields.Many2one(comodel_name='account.account', string='Loss on Disposal Account',
                                               default=_get_default_loss_account)
    gain_on_disposal_account = fields.Many2one(comodel_name='account.account', string='Depreciation Recovered Account',
                                               default=_get_default_gain_account,
                                               help='If not specified, the depreciation account for the asset is used')

    ################################################################################
    # Methods
    ################################################################################

    @api.multi
    @api.onchange('disposal_date')
    def onchange_disposal_date(self):
        self.accounting_date = self.disposal_date
        asset_id = self.env.context.get('active_id', False)
        self.asset = self.env['account.asset.asset'].browse(asset_id)
        self.asset_cost = self.asset.value
        posted_depn_lines = self.env['account.asset.depreciation.line'].search([('asset_id', '=', asset_id),
                                                                                ('move_posted_check', '=', True)])
        posted_depn = sum([x.amount for x in posted_depn_lines])
        self.asset_bv = self.asset_cost - posted_depn
        return

    @api.multi
    def move_line(self, asset, account_id, debit, credit, journal, partner, expense=False):
        move_line = {
            'name': asset.name,
            'account_id': account_id,
            'debit': debit,
            'credit': credit,
            'journal_id': journal,
            'partner_id': partner.id if partner else False,
            'analytic_account_id': False,
            'currency_id': False,
            'amount_currency': 0.0 }
        return move_line

    def get_journal_id(self):
        return self.env['account.asset.default'] \
            .search([('company_id', '=', self.env.user.company_id.id)])[0].invoice_journal.id

    @api.multi
    def button_process(self):

        if self.env.user.company_id.fiscalyear_lock_date > self.accounting_date:
            raise UserError('This accounting date is before your posting cut off date')

        journal = self.env['account.asset.default'] \
            .search([('company_id', '=', self.env.user.company_id.id)])[0].journal.id
        if not journal:
            raise UserError('No journal set up')

        asset_id = self.env.context.get('active_id', False)
        asset = self.env['account.asset.asset'].browse(asset_id)

        # check draft moves posted and delete depreciation lines not linked to a draft or posted move
        lines_to_check = [x for x in asset.depreciation_line_ids if x.move_check]
        for line in lines_to_check:
            if line.move_id and line.move_id.state == 'draft':
                raise UserError('There is a draft journal created for this asset. Please either delete or post.')

        lines_to_delete = [x for x in asset.depreciation_line_ids if not x.move_check]
        for line in lines_to_delete:
            line.unlink()

        move_lines = []
        # asset amount
        line = self.move_line(asset, asset.category_id.account_asset_id.id, 0, asset.value, journal, self.partner)

        move_lines.append((0, 0, line))
        # Accum Depn amount
        line = self.move_line(asset, asset.category_id.account_depreciation_id.id, self.asset_cost - self.asset_bv,
                              0, journal, self.partner)
        move_lines.append((0, 0, line))
        # sale proceeds
        if self.amount:
            sale_of_asset_account = self.env['account.asset.default'] \
                .search([('company_id', '=', self.env.user.company_id.id)])[0].asset_sale_suspense_account.id
            line = self.move_line(asset, sale_of_asset_account, self.amount,
                                  0, journal, self.partner, expense=True)
            move_lines.append((0, 0, line))

        capital_gain = 0.0
        depn_recovered = 0.0
        depreciation_line_amount = 0.0

        if self.asset_bv > self.amount:
            loss_on_disposal_account = self.env['account.asset.default'] \
                .search([('company_id', '=', self.env.user.company_id.id)])[0].loss_on_disposal_account.id
            line = self.move_line(asset, loss_on_disposal_account, self.asset_bv - self.amount,
                                  0, journal, self.partner, expense=True)
            move_lines.append((0, 0, line))
            depreciation_line_amount = self.asset_bv - self.amount

        elif self.amount > self.asset_bv:
            if self.amount > self.asset_cost:
                capital_gain = self.amount - self.asset_cost
                depn_recovered = self.asset_cost - self.asset_bv

            else:
                capital_gain = 0.0
                depn_recovered = self.amount - self.asset_bv
                depreciation_line_amount = 0 - depn_recovered

        if capital_gain:
            capital_gain_account = self.env['account.asset.default'] \
                .search([('company_id', '=', self.env.user.company_id.id)])[0].capital_gain_account.id
            line = self.move_line(asset, capital_gain_account, 0, capital_gain,
                                  journal, self.partner)
            move_lines.append((0, 0, line))

        if depn_recovered:
            if self.gain_on_disposal_account:
                account = self.gain_on_disposal_account.id
            else:
                account = asset.category_id.account_depreciation_expense_id.id
            line = self.move_line(asset, account, 0, depn_recovered,
                                  journal, self.partner, expense=True)
            move_lines.append((0, 0, line))

        move_vals = {
            'ref': asset.code,
            'name': asset.name + ' - Disposal',
            'date': self.accounting_date or False,
            'journal_id': journal,
            'line_ids': move_lines,
        }
        move = self.env['account.move'].sudo().create(move_vals)

        # create asset depn lines for the transaction so shows as 0

        self.env['account.asset.depreciation.line'].create({
            'asset_id': asset.id,
            'sequence': 90,
            'name': asset.name + '- Disposal',
            'amount': depreciation_line_amount,
            'remaining_value': 0.0,
            'depreciated_value': self.asset_bv + depreciation_line_amount,
            'line_type': 'loss',
            'depreciation_date': self.disposal_date,
            'move_id': move.id,
            'move_check': True
        })

        if self.amount:
            if self.amount > self.asset_cost:
                amount = self.asset_cost
            else:
                amount = self.amount
            self.env['account.asset.depreciation.line'].create({
                'asset_id': asset.id,
                'sequence': 91,
                'name': asset.name + '- Disposal',
                'amount': amount,
                'remaining_value': 0.0,
                'depreciated_value': asset.value,
                'line_type': 'sale',
                'depreciation_date': self.disposal_date,
                'move_id': move.id,
                'move_check': True
            })

        # update the asset
        asset.write({
            'state': 'close',
            'disposal_date': self.disposal_date,
            'salvage_value': 0.0})

        if self.create_invoice and self.amount:
            invoice_id = self.env['account.invoice'].create({
                'name': asset.name + ' - Disposal',
                'date_invoice': self.accounting_date,
                'origin': asset.name + ' - Disposal',
                'type': 'out_invoice',
                'reference': asset.name + ' - Disposal',
                'account_id': self.partner.property_account_receivable_id.id,
                'partner_id': self.partner.id,
                'journal_id': self.get_journal_id(),
                'currency_id': self.env.user.company_id.currency_id.id,
                'comment': False,
                'payment_term_id': self.partner.property_payment_term_id.id,
                'fiscal_position_id': self.partner.property_account_position_id.id,
                'company_id': self.env.user.company_id.id,
                'user_id': self.env.uid,
                'team_id': False,
            })
            product = self.env['account.asset.default'] \
                .search([('company_id', '=', self.env.user.company_id.id)])[0].sale_product

            taxes = self.env['account.tax'].search([('type_tax_use', '=', 'sale'),
                                                    ('amount', '=', 0.0),
                                                    ('amount_type', '=', 'percent')], order='id desc', limit=1)

            self.env['account.invoice.line'].create({
                'invoice_id': invoice_id.id,
                'name': product.name,
                'sequence': 5,
                'origin': asset.name + ' - Disposal',
                'account_id': sale_of_asset_account,
                'price_unit': self.amount,
                'quantity': 1,
                'uom_id': 1,
                'product_id': product.id,
                'invoice_line_tax_ids': [(6, 0, [x.id for x in taxes])],
            })
            invoice_id.compute_taxes()
            invoice_id._compute_amount()

        return invoice_id
