# -*- coding: utf-8 -*-

from datetime import date, datetime, timedelta

from dateutil.relativedelta import relativedelta

from odoo import api, fields, models, _
from odoo.tools import DEFAULT_SERVER_DATE_FORMAT as DF
from odoo.tools import float_compare, float_is_zero


class account_asset(models.Model):
    _inherit = "account.asset.asset"

    _order = 'date desc'
    _sql_constraints = [('unique_code', 'unique(code)', 'Code must be unique')
                        ]
    ################################################################################
    # Fields
    ################################################################################

    serial_number = fields.Char("Serial Number")
    disposal_date = fields.Date("Disposal Date")
    name = fields.Char(string='Asset Name', required=True, readonly=True,
                       states={'draft': [('readonly', False)], 'open': [('readonly', False)]})
    code = fields.Char(string='Reference', size=32, required=True, readonly=True,
                       states={'draft': [('readonly', False)], 'open': [('readonly', False)]})
    value = fields.Float(string='Gross Value', required=True, readonly=True, digits=0,
                         states={'draft': [('readonly', False)], 'open': [('readonly', False)]},
                         oldname='purchase_value')
    category_id = fields.Many2one('account.asset.category', string='Category',
                                  required=True, change_default=True,
                                  readonly=True,
                                  states={'draft': [('readonly', False)], 'open': [('readonly', False)]})
    depreciation_start_date = fields.Date(string='Depreciation Start Date',
                                          required=True, states={'draft': [('readonly', False)],
                                                                 'open': [('readonly', False)]})
    parent_id = fields.Many2one(comodel_name='account.asset.asset', string='Parent Asset')

    # added here so date from wizard correctly used on journal

    ################################################################################
    # Methods
    ################################################################################

    @api.multi
    def _compute_entries(self, date, group_entries=False):
        depreciation_ids = self.env['account.asset.depreciation.line'].search([
            ('asset_id', 'in', self.ids), ('depreciation_date', '<=', date),
            ('move_check', '=', False)])
        if group_entries:
            return depreciation_ids.create_grouped_move(date=date)
        return depreciation_ids.create_move(date=date)

    @api.multi
    def _get_last_depreciation_date(self):
        # copy and paste - if no previous depreciation entries then use the depreciation start date
        """
        @param id: ids of a account.asset.asset objects
        @return: Returns a dictionary of the effective dates of the last depreciation entry made for given asset ids. 
                If there isn't any, return the depreciation_start_date date of this asset
        """
        self.env.cr.execute("""
            SELECT a.id as id, COALESCE(MAX(m.date),a.depreciation_start_date) AS date
            FROM account_asset_asset a
            LEFT JOIN account_asset_depreciation_line rel ON (rel.asset_id = a.id)
            LEFT JOIN account_move m ON (rel.move_id = m.id)
            WHERE a.id IN %s
            GROUP BY a.id, m.date """, (tuple(self.ids),))
        result = dict(self.env.cr.fetchall())
        return result

    @api.multi
    def compute_depreciation_board(self):
        # yet another copy and paste as we want to use the depreciation start date if no depn entries posted
        # this allows the user to reset the start date
        self.ensure_one()
        posted_depn_lines_unsorted = self.depreciation_line_ids.filtered(lambda x: x.move_check)
        posted_depreciation_line_ids = posted_depn_lines_unsorted.sorted(key=lambda l: l.depreciation_date)
        unposted_depreciation_line_ids = self.depreciation_line_ids.filtered(lambda x: not x.move_check)

        # Remove old unposted depreciation lines. We cannot use unlink() with One2many field
        commands = [(2, line_id.id, False) for line_id in unposted_depreciation_line_ids]

        if self.value_residual != 0.0:
            amount_to_depr = residual_amount = self.value_residual

            if posted_depreciation_line_ids and posted_depreciation_line_ids[-1].depreciation_date:
                last_entry_depn_date = posted_depreciation_line_ids[-1].depreciation_date
                last_depreciation_date = datetime.strptime(last_entry_depn_date, DF).date()
                depreciation_date = last_depreciation_date + relativedelta(months=+self.method_period)
            else:
                depreciation_date = False

            if self.prorata:
                # if we already have some previous validated entries, starting date is last entry + method period
                if not depreciation_date:
                    depreciation_date = datetime.strptime(self._get_last_depreciation_date()[self.id], DF).date()
            else:
                # depreciation_date = 1st of January of purchase year if annual valuation, 1st of
                # purchase month in other cases
                if self.method_period >= 12:
                    asset_date = datetime.strptime(self.depreciation_start_date[:4] + '-01-01', DF).date()
                else:
                    asset_date = datetime.strptime(self.depreciation_start_date[:7] + '-01', DF).date()
                # if we already have some previous validated entries, starting date
                #  isn't 1st January but last entry + method period
                if not depreciation_date:
                    depreciation_date = asset_date

            day = depreciation_date.day
            month = depreciation_date.month
            year = depreciation_date.year
            total_days = (year % 4) and 365 or 366

            undone_dotation_number = self._compute_board_undone_dotation_nb(depreciation_date, total_days)

            for x in range(len(posted_depreciation_line_ids), undone_dotation_number):
                sequence = x + 1
                amount = self._compute_board_amount(sequence, residual_amount,
                                                    amount_to_depr, undone_dotation_number,
                                                    posted_depreciation_line_ids, total_days,
                                                    depreciation_date)

                amount = self.currency_id.round(amount)
                if float_is_zero(amount, precision_rounding=self.currency_id.rounding):
                    continue
                residual_amount -= amount
                vals = {
                    'amount': amount,
                    'asset_id': self.id,
                    'sequence': sequence,
                    'name': (self.code or '') + '/' + str(sequence),
                    'remaining_value': residual_amount,
                    'line_type': 'normal',
                    'depreciated_value': self.value - (self.salvage_value + residual_amount),
                    'depreciation_date': depreciation_date.strftime(DF),
                }
                commands.append((0, False, vals))
                # Considering Depr. Period as months
                depreciation_date = date(year, month, day) + relativedelta(months=+self.method_period)
                day = depreciation_date.day
                month = depreciation_date.month
                year = depreciation_date.year

        self.write({'depreciation_line_ids': commands})
        return True

    @api.model
    def fields_view_get(self, view_id=None, view_type='form', toolbar=False, submenu=False):
        view = super(account_asset, self).fields_view_get(view_id, view_type, toolbar, submenu)
        return view


class AccountAssetDepreciationLine(models.Model):
    _inherit = 'account.asset.depreciation.line'

    _order = 'sequence asc'

    ################################################################################
    # Fields
    ################################################################################

    category_id = fields.Many2one(comodel_name='account.asset.category', string='Asset Category',
                                  related='asset_id.category_id', store=True)

    """
    copy and paste override from core as date on journal not being set properly
    have also set admin to do posting to avoid any security issues
    """

    line_type = fields.Selection(string='Line Type', selection=
                                        [('normal', 'Depreciation'),
                                         ('loss', 'Loss/(Gain) on Disposal'),
                                         ('sale', 'Sale Proceeds'),
                                         ('impairment', 'Impairment')])
    move_state = fields.Selection(string='Move State', selection=[('draft', 'Draft'), ('posted', 'Posted')],
                                  related='move_id.state')

    """
    copy and paste - core closes an asset once fully depreciated
    """

    ################################################################################
    # Methods
    ################################################################################

    @api.multi
    def post_lines_and_close_asset(self):
        # we re-evaluate the assets to determine whether we can close them
        for line in self:
            line.log_message_when_posted()
            asset = line.asset_id
            if asset.currency_id.is_zero(asset.value_residual) and asset.disposal_date:
                asset.message_post(body=_("Document closed."))
                asset.write({'state': 'close'})

    @api.multi
    def create_move(self, post_move=True, date=False):
        autopost_depreciation_journal = self.env['account.asset.default'] \
            .search([('company_id', '=', self.env.user.company_id.id)])[0].autopost_depreciation_journal
        if autopost_depreciation_journal:
            post_move = True
        else:
            post_move = False
        created_moves = self.env['account.move']
        prec = self.env['decimal.precision'].precision_get('Account')
        for line in self:
            category_id = line.asset_id.category_id
            depreciation_date = date or self.env.context.get('depreciation_date') or line.depreciation_date \
                                or fields.Date.context_today(self)

            # check that date is still open
            period_lock_date = max(self.env.user.company_id.period_lock_date,
                                   self.env.user.company_id.fiscalyear_lock_date)
            if depreciation_date < period_lock_date:
                depreciation_date = depreciation_date + timedelta(days=1)

            company_currency = line.asset_id.company_id.currency_id
            current_currency = line.asset_id.currency_id
            amount = current_currency.compute(line.amount, company_currency)
            asset_name = line.asset_id.name + ' (%s/%s)' % (line.sequence, len(line.asset_id.depreciation_line_ids))

            expense_account_id = category_id.account_depreciation_expense_id.id

            move_line_1 = {
                'name': asset_name,
                'account_id': category_id.account_depreciation_id.id,
                'debit': 0.0 if float_compare(amount, 0.0, precision_digits=prec) > 0 else -amount,
                'credit': amount if float_compare(amount, 0.0, precision_digits=prec) > 0 else 0.0,
                'journal_id': category_id.journal_id.id,
                'partner_id': line.asset_id.partner_id.id,
                'analytic_account_id': category_id.account_analytic_id.id if category_id.type == 'sale' else False,
                'currency_id': company_currency != current_currency and current_currency.id or False,
                'amount_currency': company_currency != current_currency and -1.0 * line.amount or 0.0,
            }
            move_line_2 = {
                'name': asset_name,
                'account_id': expense_account_id,
                'credit': 0.0 if float_compare(amount, 0.0, precision_digits=prec) > 0 else -amount,
                'debit': amount if float_compare(amount, 0.0, precision_digits=prec) > 0 else 0.0,
                'journal_id': category_id.journal_id.id,
                'partner_id': line.asset_id.partner_id.id,
                'analytic_account_id': category_id.account_analytic_id.id if category_id.type == 'purchase' else False,
                'currency_id': company_currency != current_currency and current_currency.id or False,
                'amount_currency': company_currency != current_currency and line.amount or 0.0
            }

            move_vals = {
                'ref': line.asset_id.code,
                'date': depreciation_date or False,
                'journal_id': category_id.journal_id.id,
                'line_ids': [(0, 0, move_line_1), (0, 0, move_line_2)],
            }
            move = self.env['account.move'].sudo().create(move_vals)
            line.write({'move_id': move.id, 'move_check': True})
            created_moves |= move

        if post_move and created_moves:
            created_moves.post()
        return [x.id for x in created_moves]

    @api.multi
    def create_grouped_move(self, post_move=True, date=False):
        if not self.exists():
            return []
        autopost_depreciation_journal = self.env['account.asset.default'] \
            .search([('company_id', '=', self.env.user.company_id.id)])[0].autopost_depreciation_journal
        if autopost_depreciation_journal:
            post_move = True
        else:
            post_move = False
        created_moves = self.env['account.move']
        category_id = self[0].asset_id.category_id  # we can suppose that all lines have the same category
        depreciation_date = date or self.env.context.get('depreciation_date') or fields.Date.context_today(self)
        amount = 0.0
        for line in self:
            # Sum amount of all depreciation lines
            company_currency = line.asset_id.company_id.currency_id
            current_currency = line.asset_id.currency_id
            amount += current_currency.compute(line.amount, company_currency)

        name = category_id.name + _(' (grouped)')
        move_line_1 = {
            'name': name,
            'account_id': category_id.account_depreciation_id.id,
            'debit': 0.0,
            'credit': amount,
            'journal_id': category_id.journal_id.id,
            'analytic_account_id': category_id.account_analytic_id.id if category_id.type == 'sale' else False,
        }
        move_line_2 = {
            'name': name,
            'account_id': category_id.account_depreciation_expense_id.id,
            'credit': 0.0,
            'debit': amount,
            'journal_id': category_id.journal_id.id,
            'analytic_account_id': category_id.account_analytic_id.id if category_id.type == 'purchase' else False
        }
        move_vals = {
            'ref': category_id.name,
            'date': depreciation_date or False,
            'journal_id': category_id.journal_id.id,
            'line_ids': [(0, 0, move_line_1), (0, 0, move_line_2)],
        }
        move = self.env['account.move'].sudo().create(move_vals)
        self.write({'move_id': move.id, 'move_check': True})
        created_moves |= move

        if post_move and created_moves:
            for line in self:
                line.log_message_when_posted()
            created_moves.post()
        return [x.id for x in created_moves]
